"""
Flask-based Currency Conversion API for Namecheap Shared Hosting
----------------------------------------------------------------
This version is fully compatible with Passenger WSGI hosting.
It implements:

✔ API keys (free + paid)
✔ Rate limiting (per minute + per day)
✔ Usage tracking & monthly credit resets
✔ Paystack subscription upgrade
✔ Admin endpoints (generate/revoke/list/usage)
✔ Lagos time for created_at & upgraded_at
✔ No FastAPI imports anywhere
✔ key_prefix (no key_hash)
"""

from __future__ import annotations
import os
import time
from typing import Dict, Any, Optional

from flask import Flask, request, jsonify

from auth import (
    validate_api_key,
    generate_api_key,
    generate_free_key,
    revoke_api_key,
    load_keys,
    upgrade_key_tier_by_hash,
    charge_call_for_key,
    SECONDS_PER_MONTH,
    now_wat_iso,
)
from rate_limit import allow_request
from utils import validate_currency, fetch_rates
from billing import initialize_transaction, verify_paystack_signature, extract_webhook_info

from dotenv import load_dotenv
load_dotenv()

# ---------------------------------------------------------------------------
# Admin Token
# ---------------------------------------------------------------------------

ADMIN_FILE = "admin_token.txt"

def _load_or_create_admin_token() -> str:
    env_token = os.getenv("ADMIN_TOKEN")
    if env_token:
        return env_token

    if os.path.exists(ADMIN_FILE):
        with open(ADMIN_FILE, "r", encoding="utf-8") as fh:
            tok = fh.read().strip()
            if tok:
                return tok

    import secrets
    tok = "admtok_" + secrets.token_hex(16)
    with open(ADMIN_FILE, "w", encoding="utf-8") as fh:
        fh.write(tok)

    print("\n==============================\n"
          " NEW ADMIN TOKEN GENERATED\n"
          f" TOKEN: {tok}\n"
          " SAVED TO: admin_token.txt\n"
          "==============================\n")
    return tok

ADMIN_TOKEN = _load_or_create_admin_token()

# ---------------------------------------------------------------------------
# Plans
# ---------------------------------------------------------------------------

PLANS = [
    {
        "code": "BASIC_MONTHLY",
        "name": "Basic Monthly",
        "price_naira": 2000,
        "currency": "NGN",
        "requests_per_minute": 100,
        "requests_per_day": 10_000,
        "monthly_credits": 5000,
    },
    {
        "code": "PRO_MONTHLY",
        "name": "Pro Monthly",
        "price_naira": 5000,
        "currency": "NGN",
        "requests_per_minute": 300,
        "requests_per_day": 50_000,
        "monthly_credits": 20000,
    },
]

FREE_REQUESTS_PER_MINUTE = 10
FREE_REQUESTS_PER_DAY = 200
FREE_TIER_MONTHLY_CREDITS = 200

DEFAULT_PAID_REQUESTS_PER_MINUTE = 100
DEFAULT_PAID_REQUESTS_PER_DAY = 10_000

# ---------------------------------------------------------------------------
# Flask App Init
# ---------------------------------------------------------------------------

app = Flask(__name__)

# ---------------------------------------------------------------------------
# Request Logging
# ---------------------------------------------------------------------------

@app.before_request
def _req_start():
    request._start_time = time.time()

@app.after_request
def _req_end(resp):
    try:
        ms = round((time.time() - request._start_time) * 1000, 2)
        print(f"[{request.method}] {request.path} {resp.status_code} {ms}ms")
    except:
        pass
    return resp

# ---------------------------------------------------------------------------
# Utility Helpers
# ---------------------------------------------------------------------------

def abort_json(code: int, msg: str):
    r = jsonify({"detail": msg})
    r.status_code = code
    return r

def require_admin():
    tok = request.headers.get("X-ADMIN-TOKEN") or request.args.get("admin_token")
    if tok != ADMIN_TOKEN:
        return abort_json(403, "Invalid admin token")
    return None

def resolve_limits(key_record: Dict[str, Any]):
    tier = key_record.get("tier", "paid")
    plan_code = key_record.get("plan_code")

    if tier == "free":
        return {
            "per_minute": FREE_REQUESTS_PER_MINUTE,
            "per_day": FREE_REQUESTS_PER_DAY,
            "plan": None,
        }

    if plan_code:
        plan = next((p for p in PLANS if p["code"] == plan_code), None)
        if plan:
            return {
                "per_minute": plan["requests_per_minute"],
                "per_day": plan["requests_per_day"],
                "plan": plan,
            }

    return {
        "per_minute": DEFAULT_PAID_REQUESTS_PER_MINUTE,
        "per_day": DEFAULT_PAID_REQUESTS_PER_DAY,
        "plan": None,
    }

def require_api_key():
    raw = request.headers.get("X-API-KEY") or request.args.get("api_key")
    if not raw:
        raise RuntimeError(abort_json(401, "Missing API Key"))

    key_record = validate_api_key(raw)
    limits = resolve_limits(key_record)

    allow_request(
        key_prefix=key_record["key_prefix"],
        per_minute=limits["per_minute"],
        per_day=limits["per_day"],
    )

    charge_call_for_key(
        key_record,
        free_monthly_credits=FREE_TIER_MONTHLY_CREDITS,
        plans=PLANS,
    )

    return key_record

# ---------------------------------------------------------------------------
# Routes
# ---------------------------------------------------------------------------

@app.route("/health", methods=["GET"])
def health():
    return jsonify({"status": "ok"})

# ---------------- AUTH ----------------

@app.route("/auth/free-key", methods=["GET", "POST"])
def free_key():
    raw = generate_free_key()
    return jsonify({
        "tier": "free",
        "api_key": raw,
        "note": "Free keys have limited quota. Upgrade for more usage.",
    })

@app.route("/me/usage", methods=["GET"])
def me_usage():
    try:
        key = require_api_key()
    except RuntimeError as e:
        return e.args[0]

    limits = resolve_limits(key)

    return jsonify({
        "tier": key.get("tier"),
        "plan_code": key.get("plan_code"),
        "limits": limits,
        "credits": key.get("credits"),
        "credits_reset_at": key.get("credits_reset_at"),
        "created_at": key.get("created_at"),
        "upgraded_at": key.get("upgraded_at"),
    })

# ---------------- RATE ----------------

@app.route("/rate", methods=["GET"])
def api_rate():
    try:
        require_api_key()
    except RuntimeError as e:
        return e.args[0]

    fc = request.args.get("from")
    tc = request.args.get("to")
    if not fc or not tc:
        return abort_json(400, "from & to required")

    fc_l = validate_currency(fc)
    tc_l = validate_currency(tc)

    rates = fetch_rates(fc_l)
    if tc_l not in rates:
        return abort_json(400, f"Unsupported currency {tc.upper()}")

    return jsonify({
        "from": fc.upper(),
        "to": tc.upper(),
        "rate": rates[tc_l],
    })

# ---------------- CONVERT ----------------

@app.route("/convert", methods=["GET"])
def api_convert():
    try:
        require_api_key()
    except RuntimeError as e:
        return e.args[0]

    fc = request.args.get("from")
    tc = request.args.get("to")
    amt = request.args.get("amount")

    if not fc or not tc or amt is None:
        return abort_json(400, "from, to, amount required")

    try:
        amount = float(amt)
    except:
        return abort_json(400, "amount must be numeric")

    if amount < 0:
        return abort_json(400, "amount must be >= 0")

    fc_l = validate_currency(fc)
    tc_l = validate_currency(tc)

    rates = fetch_rates(fc_l)
    if tc_l not in rates:
        return abort_json(400, "Unsupported target currency")

    rate = rates[tc_l]
    return jsonify({
        "from": fc.upper(),
        "to": tc.upper(),
        "amount": amount,
        "rate": rate,
        "converted_amount": round(amount * rate, 4),
    })

# ---------------- PLANS ----------------

@app.route("/plans", methods=["GET"])
def list_plans():
    return jsonify({"plans": PLANS})

@app.route("/billing/upgrade", methods=["POST"])
def billing_upgrade():
    try:
        key = require_api_key()
    except RuntimeError as e:
        return e.args[0]

    data = request.get_json(silent=True) or {}
    email = data.get("email")
    plan_code = data.get("plan_code")

    if not email or not plan_code:
        return abort_json(400, "email + plan_code required")

    plan = next((p for p in PLANS if p["code"] == plan_code), None)
    if not plan:
        return abort_json(400, "Invalid plan_code")

    auth_url, reference = initialize_transaction(
        email=email,
        amount_naira=plan["price_naira"],
        api_key_prefix=key["key_prefix"],
        plan_code=plan_code,
    )

    return jsonify({
        "authorization_url": auth_url,
        "reference": reference,
        "plan": plan,
    })

# ---------------- PAYSTACK WEBHOOK ----------------

@app.route("/paystack/webhook", methods=["POST"])
def paystack_webhook():
    body = request.get_data()
    sig = request.headers.get("x-paystack-signature")

    if not verify_paystack_signature(body, sig):
        return abort_json(400, "Invalid signature")

    event_type, key_prefix, plan_code = extract_webhook_info(body)

    if event_type in ("charge.success", "subscription.enable", "subscription.create"):
        if key_prefix and plan_code:
            plan = next((p for p in PLANS if p["code"] == plan_code), None)
            if plan:
                upgrade_key_tier_by_hash(
                    key_hash=key_prefix,
                    new_tier="paid",
                    plan=plan,
                    free_monthly_credits=FREE_TIER_MONTHLY_CREDITS,
                )

    if event_type in ("subscription.disable", "invoice.failed", "invoice.create_failed"):
        if key_prefix:
            upgrade_key_tier_by_hash(
                key_hash=key_prefix,
                new_tier="free",
                plan=None,
                free_monthly_credits=FREE_TIER_MONTHLY_CREDITS,
            )

    return jsonify({"status": "ok"})

# ---------------- ADMIN ----------------

@app.route("/admin/generate-key", methods=["POST"])
def admin_generate_key():
    adm = require_admin()
    if adm: return adm
    data = request.get_json(silent=True) or {}
    owner = data.get("owner")
    if not owner:
        return abort_json(400, "owner required")
    raw = generate_api_key(owner)
    return jsonify({"owner": owner, "api_key": raw})

@app.route("/admin/revoke-key", methods=["POST"])
def admin_revoke_key():
    adm = require_admin()
    if adm: return adm
    data = request.get_json(silent=True) or {}
    key = data.get("api_key")
    if not key:
        return abort_json(400, "api_key required")
    revoke_api_key(key)
    return jsonify({"status": "ok"})

@app.route("/admin/keys", methods=["GET"])
def admin_keys():
    adm = require_admin()
    if adm: return adm
    data = load_keys()
    return jsonify({"keys": data.get("keys", [])})

@app.route("/admin/usage", methods=["GET"])
def admin_usage():
    adm = require_admin()
    if adm: return adm

    data = load_keys()
    now = time.time()
    modified = False
    usage = []

    for r in data.get("keys", []):
        tier = r.get("tier")
        plan_code = r.get("plan_code")

        if tier == "free" or not plan_code:
            per_min = FREE_REQUESTS_PER_MINUTE
            per_day = FREE_REQUESTS_PER_DAY
        else:
            plan = next((p for p in PLANS if p["code"] == plan_code), None)
            per_min = plan["requests_per_minute"] if plan else 100
            per_day = plan["requests_per_day"] if plan else 10000

        # Reset credits
        reset_at = r.get("credits_reset_at")
        if reset_at and now >= reset_at:
            if tier == "free":
                r["credits"] = FREE_TIER_MONTHLY_CREDITS
            else:
                plan = next((p for p in PLANS if p["code"] == plan_code), None)
                r["credits"] = plan["monthly_credits"] if plan else 20000
            r["credits_reset_at"] = now + SECONDS_PER_MONTH
            modified = True

        usage.append({
            "owner": r.get("owner"),
            "tier": tier,
            "plan_code": plan_code,
            "remaining_credits": r.get("credits"),
            "credits_reset_at": r.get("credits_reset_at"),
            "rate_limits": {"per_minute": per_min, "per_day": per_day},
            "key_prefix": r.get("key_prefix"),
        })

    if modified:
        from auth import save_keys
        save_keys(data)

    return jsonify({"usage": usage})


if __name__ == "__main__":
    app.run(debug=True, port=8000)