import json
import os
import secrets
from datetime import datetime, timedelta
import pytz
from flask import abort, jsonify

KEYS_FILE = os.path.join(os.path.dirname(__file__), "keys.json")
TIMEZONE = pytz.timezone("Africa/Lagos")

# 30 days in seconds
SECONDS_PER_MONTH = 30 * 24 * 60 * 60


def now_wat():
    """Return timezone-aware WAT datetime."""
    return datetime.now(TIMEZONE)


def now_wat_iso():
    """Return ISO datetime in WAT without +01:00 suffix."""
    return now_wat().strftime("%Y-%m-%d %H:%M:%S.%f")


def load_keys():
    if not os.path.exists(KEYS_FILE):
        return {"keys": []}
    with open(KEYS_FILE, "r") as f:
        return json.load(f)


def save_keys(data):
    with open(KEYS_FILE, "w") as f:
        json.dump(data, f, indent=4)


def validate_api_key(raw_key: str):
    if not raw_key:
        abort(401, description="Missing API key")

    prefix = raw_key[:8]
    data = load_keys()

    for record in data["keys"]:
        if record["key_prefix"] == prefix:
            if not record.get("active", True):
                abort(403, description="API key is inactive")
            return record

    abort(401, description="Invalid API key")


def _create_key_record(owner, tier, monthly_credits):
    prefix = secrets.token_hex(4)

    return {
        "owner": owner,
        "active": True,
        "tier": tier,
        "plan_code": None,
        "created_at": now_wat_iso(),
        "credits": monthly_credits,
        "credits_reset_at": (now_wat() + timedelta(days=30)).timestamp(),
        "key_prefix": prefix,
    }


def generate_api_key(owner: str):
    record = _create_key_record(owner, "paid", monthly_credits=20000)

    data = load_keys()
    data["keys"].append(record)
    save_keys(data)

    return record["key_prefix"] + secrets.token_hex(16)


def generate_free_key():
    record = _create_key_record("free_user", "free", monthly_credits=200)

    data = load_keys()
    data["keys"].append(record)
    save_keys(data)

    return record["key_prefix"] + secrets.token_hex(16)


def revoke_api_key(api_key: str):
    prefix = api_key[:8]
    data = load_keys()

    for record in data["keys"]:
        if record["key_prefix"] == prefix:
            record["active"] = False
            save_keys(data)
            return

    abort(404, description="API key not found")


def upgrade_key_tier_by_hash(key_hash, new_tier, plan=None, free_monthly_credits=200):
    prefix = key_hash[:8]
    data = load_keys()

    for record in data["keys"]:
        if record["key_prefix"] == prefix:
            record["tier"] = new_tier
            record["plan_code"] = plan["code"] if plan else None
            if new_tier == "free":
                record["credits"] = free_monthly_credits
            else:
                record["credits"] = plan["monthly_credits"] if plan else 20000
            record["credits_reset_at"] = (now_wat() + timedelta(days=30)).timestamp()
            record["upgraded_at"] = now_wat_iso()

            save_keys(data)
            return

    abort(404, description="API key not found")


def charge_call_for_key(key_record, free_monthly_credits, plans):
    """Subtract 1 credit per call."""

    now_ts = now_wat().timestamp()

    if now_ts >= key_record["credits_reset_at"]:
        # Reset credits
        plan = None
        if key_record["plan_code"]:
            plan = next((p for p in plans if p["code"] == key_record["plan_code"]), None)

        if key_record["tier"] == "free":
            key_record["credits"] = free_monthly_credits
        else:
            key_record["credits"] = plan["monthly_credits"] if plan else 20000

        key_record["credits_reset_at"] = (now_wat() + timedelta(days=30)).timestamp()

        data = load_keys()
        save_keys(data)

    # Deduct credit
    if key_record["credits"] is not None:
        if key_record["credits"] <= 0:
            abort(402, description="No credits remaining")

        key_record["credits"] -= 1

        data = load_keys()
        save_keys(data)