"""
Rate limiting logic for the currency conversion service.

Tracks requests per API key prefix and enforces per‑minute and per‑day limits.
Designed to be framework-agnostic; callers handle how errors are returned.
"""

from __future__ import annotations

import time
from typing import Dict, List


class RateLimitError(Exception):
    """Raised when a client exceeds the configured rate limits."""

    def __init__(self, message: str, status_code: int = 429):
        super().__init__(message)
        self.message = message
        self.status_code = status_code


# In‑memory store of request timestamps per key prefix
request_timestamps: Dict[str, List[float]] = {}


def allow_request(*, key_prefix: str, per_minute: int | None, per_day: int | None) -> None:
    """
    Record a request for the given key_prefix and enforce rate limits.

    Args:
        key_prefix: The stable identifier for the API key (first 8 chars).
        per_minute: Maximum allowed requests per rolling 60‑second window.
        per_day: Maximum allowed requests per rolling 24‑hour window.

    Raises:
        RateLimitError: If the caller has exceeded either the per‑minute
        or per‑day limit.
    """
    now = time.time()
    if key_prefix not in request_timestamps:
        request_timestamps[key_prefix] = []

    # Only keep last 24h of timestamps
    timestamps = [t for t in request_timestamps[key_prefix] if now - t < 86400]
    last_minute_count = len([t for t in timestamps if now - t < 60])

    if per_minute is not None and last_minute_count >= per_minute:
        raise RateLimitError(
            "Rate limit exceeded (per minute). Please slow down.",
            status_code=429,
        )

    if per_day is not None and len(timestamps) >= per_day:
        raise RateLimitError(
            "Rate limit exceeded (per day). Please try again later.",
            status_code=429,
        )

    timestamps.append(now)
    request_timestamps[key_prefix] = timestamps
