"""Paystack billing integration for subscription management."""

from __future__ import annotations

from typing import Tuple, Optional, Dict
import json
import os
import hmac
import hashlib

import requests

PAYSTACK_SECRET_KEY = os.getenv("PAYSTACK_SECRET_KEY", "")
PAYSTACK_BASE_URL = "https://api.paystack.co"


def _headers() -> Dict[str, str]:
    if not PAYSTACK_SECRET_KEY:
        raise RuntimeError("PAYSTACK_SECRET_KEY is not set")
    return {
        "Authorization": f"Bearer {PAYSTACK_SECRET_KEY}",
        "Content-Type": "application/json",
    }


def initialize_transaction(
    *,
    email: str,
    amount_naira: int,
    api_key_prefix: str,
    plan_code: str,
) -> Tuple[str, str]:
    """Initialise a Paystack transaction for subscription upgrade.

    Returns (authorization_url, reference).
    """
    payload = {
        "email": email,
        "amount": amount_naira * 100,  # Paystack expects kobo
        "metadata": {
            "api_key_prefix": api_key_prefix,
            "plan_code": plan_code,
        },
    }
    url = f"{PAYSTACK_BASE_URL}/transaction/initialize"
    r = requests.post(url, headers=_headers(), data=json.dumps(payload), timeout=10)
    if not r.ok:
        raise RuntimeError(f"Paystack init failed: {r.status_code} {r.text}")
    data = r.json().get("data") or {}
    return data.get("authorization_url"), data.get("reference")


def verify_paystack_signature(body: bytes, signature: Optional[str]) -> bool:
    if not signature or not PAYSTACK_SECRET_KEY:
        return False
    computed = hmac.new(
        PAYSTACK_SECRET_KEY.encode("utf-8"),
        body,
        hashlib.sha512,
    ).hexdigest()
    return hmac.compare_digest(computed, signature)


def extract_webhook_info(body: bytes) -> Tuple[Optional[str], Optional[str], Optional[str]]:
    """Extract (event_type, api_key_prefix, plan_code) from webhook payload."""
    try:
        data = json.loads(body.decode("utf-8"))
    except Exception:
        return None, None, None

    event_type = data.get("event")
    event_data = data.get("data") or {}
    metadata = event_data.get("metadata") or {}

    api_key_prefix = metadata.get("api_key_prefix")
    plan_code = metadata.get("plan_code") or metadata.get("plan")
    return event_type, api_key_prefix, plan_code
